
// Digital Audio Delay firmware for dsPIC33FJ64GP802 or dsPIC33FJ128GP802
// (c) Silicon Chip, November 2011 (by Nicholas Vinen)
// Compile with -O3 -funroll-loops -fomit-frame-pointer

#include "p33Fxxxx.h"
#include "Simple EEPROM Emulation.h"
#include "main.h"
#include "sram.h"
#include "infrared.h"
#include <string.h>

#ifndef __DELAY_H
 #define FOSC  80000000LL
 #define FCY   (FOSC/2)  // MCU is running at FCY MIPS
 #define delay_us(x) __delay32(((x*FCY)/1000000L)) // delays x us
 #define delay_ms(x) __delay32(((x*FCY)/1000L))  // delays x ms
 #define __DELAY_H 1
 #include <libpic30.h>
#endif


//// Device configuration registers ////

_FGS(GWRP_OFF & GCP_OFF); // we don't want to write protect the flash
_FOSCSEL(FNOSC_FRC); // start up initially without the PLL but instead using the internal fast RC oscillator
_FOSC(FCKSM_CSECMD & OSCIOFNC_ON & POSCMD_EC);
_FWDT(FWDTEN_OFF); // not using the watchdog timer


//// Program constants ////

#define BUFFER_SIZE_WORDS 128
#define NUM_INCOMING_BUFFERS 4
#define NUM_OUTGOING_BUFFERS 4
#define BUFFERS_IN_SRAM ((unsigned short)(512UL * 1024UL / 7 / (BUFFER_SIZE_WORDS/8) - 1))
#define PREAMBLE_X 0xE200
#define PREAMBLE_Y 0xE400
#define PREAMBLE_Z 0xE800


//// Buffers ////

unsigned short RxBufferA[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
unsigned short RxBufferB[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
unsigned short TxBufferA[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
unsigned short TxBufferB[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
unsigned short incoming[NUM_INCOMING_BUFFERS][BUFFER_SIZE_WORDS], outgoing[NUM_OUTGOING_BUFFERS][BUFFER_SIZE_WORDS];
unsigned short block_states[(BUFFERS_IN_SRAM + 15) / 16];
unsigned char DecodedDataBuffer[7*(BUFFER_SIZE_WORDS/8)];
unsigned short RxHoldingBuffer[BUFFER_SIZE_WORDS];


//// Tables ////

unsigned char biphase_decode_low[256], biphase_decode_high[256];
unsigned short biphase_encode_8[256];
unsigned char biphase_encode_4[16];
unsigned short polarity_table[2] = { 0x0000, 0xFFFF };


//// Global variables ////

volatile unsigned char locked, semilocked, incoming_count, outgoing_count, write_to_flash, bypass_mode, spdif_rate_change;
unsigned char lock_timer, block_offset, frame_offset, incoming_head, incoming_tail, outgoing_head, outgoing_tail, underflow, overflow, delay_changed_counter_low, partial_buffer, dontskip, last_bit = 2;
unsigned short polarity, delay_in_ms, last_spdif_timer[4], last_spdif_timer_index, delay_changed_counter_high, stored_delay_in_ms, last_received_word, current_bit_offset;


//// Initialisation functions ////

static void setup_clock() {
  // Fosc=40MHz (Xtal = 8MHz, PLLPRE=0 [N1=2], PLLPOST=0 [N2=2], PLLDIV=39 [M=40])
  PLLFBD = 38;
  CLKDIVbits.PLLPRE = 0;
#ifdef SLOW_CLOCK
  CLKDIVbits.PLLPOST = 1;
#else
  CLKDIVbits.PLLPOST = 0;
#endif
  __builtin_write_OSCCONH(0x03); // switch to PLL
  __builtin_write_OSCCONL(0x01);
}

static void generate_biphase_tables() {
  unsigned short i, j, s;

  for( i = 0; i < 256; ++i ) {
    unsigned char val = 0, temp = i;
    for( j = 0; j < 4; ++j ) {
      val >>= 1;
      if( (temp&192) == 64 || (temp&192) == 128 )
        val |= 8;
      temp <<= 2;
    }
    biphase_decode_low[i] = val;
    biphase_decode_high[i] = val<<4;

    s = 0;
    for( j = 0; j < 8; ++j ) {
      s <<= 2;
      if( i&(1<<j) ) {
        s |= s & 4 ? 1 : 2;
      } else {
        s |= s & 4 ? 0 : 3;
      }
    }
    biphase_encode_8[i] = s;
    if( i < 16 )
      biphase_encode_4[i] = s>>8;
  }
}

void setup_dci() {
  // RB2/RP2/AN4 = data input
  TRISBbits.TRISB2 = 1;
  AD1PCFGLbits.PCFG4 = 1;
  RPINR24bits.CSDIR = 2;

  // RB0/RP0/AN2 = data output
  TRISBbits.TRISB0 = 0;
  AD1PCFGLbits.PCFG2 = 1;
  RPOR0bits.RP0R = 13;

  // RB1/RP1/AN3 = clock input
  TRISBbits.TRISB1 = 1;
  AD1PCFGLbits.PCFG3 = 1;
  RPINR24bits.CSCKR = 1;

  // set up DCI
  TSCON = 1;
  RSCON = 1;

  DMA0CONbits.DIR = 1;
  DMA0CONbits.MODE = 2;
  DMA0REQbits.IRQSEL = 0x3c;
  DMA0STA = __builtin_dmaoffset(TxBufferA);
  DMA0STB = __builtin_dmaoffset(TxBufferB);
  DMA0PAD = (volatile unsigned int) &TXBUF0;
  DMA0CNT = sizeof(TxBufferA)/2-1;
  IFS0bits.DMA0IF = 0;
  IEC0bits.DMA0IE = 1;
  DMA0CONbits.CHEN = 1;

  DMA1CONbits.DIR = 0;
  DMA1CONbits.MODE = 2;
  DMA1REQbits.IRQSEL = 0x3c;
  DMA1STA = __builtin_dmaoffset(RxBufferA);
  DMA1STB = __builtin_dmaoffset(RxBufferB);
  DMA1PAD = (volatile unsigned int) &RXBUF0;
  DMA1CNT = sizeof(RxBufferA)/2-1;
  IFS0bits.DMA1IF = 0;
  IEC0bits.DMA1IE = 1;
  DMA1CONbits.CHEN = 1;

  DCICON1 = 0;
  DCICON2 = 0;
  DCICON1bits.DCIEN = 1;
  DCICON1bits.CSCKD = 1;
  DCICON1bits.CSCKE = 1;
  DCICON1bits.DJST = 1;
  DCICON2bits.WS = 16-1;
  DCICON3bits.BCG = 10;
  IEC3bits.DCIIE = 0;
}

void setup_spdif_in_timer() {
  T1CONbits.TON = 1;
}


//// Biphase data handling functions ////

static inline unsigned char getval8(unsigned char val) {
    unsigned char ret = val ^ polarity;
    polarity = polarity_table[ret&1];
    return ret;
}

static inline unsigned short getval16(unsigned short val) {
    unsigned short ret = val ^ polarity;
    polarity = polarity_table[ret&1];
    return ret;
}

void decode_data(unsigned char* dst, const unsigned char* src) {
  unsigned char i;
  for( i = 0; i < BUFFER_SIZE_WORDS/8; ++i ) {
    dst[0] = biphase_decode_low[src[0]]|biphase_decode_high[src[8]];
    dst[1] = biphase_decode_low[src[3]]|biphase_decode_high[src[2]];
    dst[2] = biphase_decode_low[src[5]]|biphase_decode_high[src[4]];
    dst[3] = biphase_decode_low[src[7]]|biphase_decode_high[src[6]];
    dst[4] = biphase_decode_low[src[11]]|biphase_decode_high[src[10]];
    dst[5] = biphase_decode_low[src[13]]|biphase_decode_high[src[12]];
    dst[6] = biphase_decode_low[src[15]]|biphase_decode_high[src[14]];
    dst += 7;
    src += 16;
  }
}

void encode_data(unsigned char* dst, const unsigned char* src, unsigned char frame_start) {
  unsigned char i;
  unsigned short preambleL = frame_start ? PREAMBLE_Z : PREAMBLE_X, preambleR = PREAMBLE_Y;
  for( i = 0; i < BUFFER_SIZE_WORDS/8; ++i ) {
    ((unsigned short*)dst)[0] = getval16(preambleL|biphase_encode_4[src[0]&0x0F]);
    ((unsigned short*)dst)[1] = getval16(biphase_encode_8[src[1]]);
    ((unsigned short*)dst)[2] = getval16(biphase_encode_8[src[2]]);
    ((unsigned short*)dst)[3] = getval16(biphase_encode_8[src[3]]);
    ((unsigned short*)dst)[4] = getval16(preambleR|biphase_encode_4[src[0]>>4]);
    ((unsigned short*)dst)[5] = getval16(biphase_encode_8[src[4]]);
    ((unsigned short*)dst)[6] = getval16(biphase_encode_8[src[5]]);
    ((unsigned short*)dst)[7] = getval16(biphase_encode_8[src[6]]);
    preambleL = PREAMBLE_X;
    dst += 16;
    src += 7;
  }
}

unsigned char check_biphase(const unsigned short* a, unsigned short words) {
  while( words-- ) {
    if( (a[0]>>15) == last_bit || (((a[0]>>7)&3) != 1 && ((a[0]>>7)&3) != 2) )
      return 0;
    last_bit = a[0]&1;
    ++a;
  }
  return 1;
}


//// Utility functions ////

static unsigned short abs(signed short val) {
  if( val < 0 )
    return -val;
  else
    return val;
}

static unsigned char memory_ok() {
  unsigned short i;
  unsigned char j;

  LATBbits.LATB15 = 1;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, 0x55 + i + j);
    } while( j );
  } while( i );

  LATBbits.LATB15 = 0;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != (unsigned char)(0x55 + i + j) )
        return 0;
    } while( j );
  } while( i );

  LATBbits.LATB15 = 1;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, ~0x55 + i + j);
    } while( j );
  } while( i );

  LATBbits.LATB15 = 0;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != (unsigned char)(~0x55 + i + j) )
        return 0;
    } while( j );
  } while( i );

  LATBbits.LATB15 = 1;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, 0);
    } while( j );
  } while( i );

  LATBbits.LATB15 = 0;
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != 0 )
        return 0;
    } while( j );
  } while( i );

  LATBbits.LATB15 = 1;
  return 1;
}

static unsigned short calculate_delay_blocks(unsigned short delay_in_ms) {
  unsigned long accum = 0;
  unsigned short avg_clocks, ret;
  unsigned char i;

  for( i = 0; i < 4; ++i )
    accum += last_spdif_timer[i];
  avg_clocks = accum/4;
  // timer rate is 20MHz so each block stored is (avg_clocks * 50ns)
  // so each ms requires a delay of 40,000 clocks
  ret = delay_in_ms * 40000UL / avg_clocks;
  if( ret > BUFFERS_IN_SRAM - 4 )
    ret = BUFFERS_IN_SRAM - 4;
  return ret;
}


//// DMA Interrupt handlers and helpers ////

void __attribute__((__interrupt__,no_auto_psv)) _DMA0Interrupt(void) {
  unsigned short* buf = DMACS1bits.PPST0 ? TxBufferA : TxBufferB;
  IFS0bits.DMA0IF = 0; // Clear the DMA0 Interrupt Flag

  if( DMA0CNT != sizeof(TxBufferA)/2-1 ) {
    asm volatile("disi #16");
    DMA0CONbits.CHEN = 0;
    DMA1CONbits.CHEN = 0;
    DMA0CNT = sizeof(TxBufferA)/2-1;
    DMA1CNT = sizeof(TxBufferA)/2-1;
    DMA0CONbits.CHEN = 1;
    DMA1CONbits.CHEN = 1;
  }
  if( !bypass_mode ) {
    if( outgoing_count > 0 ) {
      memcpy(buf, outgoing[outgoing_head], sizeof(TxBufferA));
      if( ++outgoing_head == NUM_OUTGOING_BUFFERS )
        outgoing_head = 0;
      --outgoing_count;
    } else {
      underflow = 1;
    }
  }
}

static unsigned short calc_bit_offset(unsigned short* buf) {
  unsigned char i;
  unsigned short value;

  for( i = 0; i < 16; ++i ) {
    if( i == 0 )
      value = buf[0]&0xFF00;
    else
      value = ((buf[0]>>i)|(last_received_word<<(16-i)))&0xFF00;
    if( value == PREAMBLE_Z || value == (PREAMBLE_Z^0xFF00) ||
        value == PREAMBLE_X || value == (PREAMBLE_X^0xFF00) ) {
      return i;
    }
  }

  if( dontskip ) {
    --dontskip;
  } else {
    asm volatile("disi #16");
    DMA0CONbits.CHEN = 0;
    DMA1CONbits.CHEN = 0;
    --DMA0CNT;
    --DMA1CNT;
    DMA0CONbits.CHEN = 1;
    DMA1CONbits.CHEN = 1;
    dontskip = 3;
  }
  return 0xFFFF;
}

void __attribute__((__interrupt__,no_auto_psv)) _DMA1Interrupt(void) {
  unsigned short* buf = DMACS1bits.PPST0 ? RxBufferA : RxBufferB;
  unsigned short timer = TMR1;
  unsigned short first_word;
  TMR1 = 0;

  memcpy(RxHoldingBuffer, buf, sizeof(RxHoldingBuffer));
  IFS0bits.DMA1IF = 0; // Clear the DMA0 Interrupt Flag
  if( bypass_mode ) {
    if( buf == RxBufferA )
      memcpy(TxBufferA, RxHoldingBuffer, sizeof(TxBufferA));
    else
      memcpy(TxBufferB, RxHoldingBuffer, sizeof(TxBufferB));
  } else {
    if( locked ) {
      if( abs(last_spdif_timer[last_spdif_timer_index] - timer) > 10 ) {
	    if( spdif_rate_change < 3 )
          ++spdif_rate_change;
      } else {
        if( spdif_rate_change < 3 )
          spdif_rate_change = 0;
      } 

      if( incoming_count+partial_buffer < NUM_INCOMING_BUFFERS ) {
        if( frame_offset == 0 ) {
          if( current_bit_offset == 0 ) {
            memcpy(incoming[incoming_tail], RxHoldingBuffer, sizeof(incoming[incoming_tail]));
          } else {
            unsigned short* dest = incoming[incoming_tail];
            unsigned short* src = RxHoldingBuffer;
            unsigned short words = BUFFER_SIZE_WORDS-1;
            dest[0] = ((src[0]>>current_bit_offset)|(last_received_word<<(16-current_bit_offset)));
            ++dest;
            ++src;
            do {
              dest[0] = ((src[0]>>current_bit_offset)|(src[-1]<<(16-current_bit_offset)));
              ++dest;
              ++src;
            } while( --words );
          }
        } else {
          unsigned char new_tail;

          if( partial_buffer ) {
            if( current_bit_offset == 0 ) {
              memcpy(incoming[incoming_tail] + BUFFER_SIZE_WORDS - frame_offset, RxHoldingBuffer, frame_offset * sizeof(unsigned short));
            } else {
              unsigned short* dest = incoming[incoming_tail] + BUFFER_SIZE_WORDS - frame_offset;
              unsigned short* src = RxHoldingBuffer;
              unsigned short words = frame_offset - 1;
              dest[0] = ((src[0]>>current_bit_offset)|(last_received_word<<(16-current_bit_offset)));
              ++dest;
              ++src;
              do {
                dest[0] = ((src[0]>>current_bit_offset)|(src[-1]<<(16-current_bit_offset)));
                ++dest;
                ++src;
              } while( --words );
            }
          }

          new_tail = incoming_tail + 1;
          if( new_tail == NUM_INCOMING_BUFFERS )
            new_tail = 0;
          if( current_bit_offset == 0 ) {
            memcpy(incoming[new_tail], RxHoldingBuffer + frame_offset, (BUFFER_SIZE_WORDS - frame_offset) * sizeof(unsigned short));
          } else {
            unsigned short* dest = incoming[new_tail];
            unsigned short* src = RxHoldingBuffer + frame_offset;
            unsigned short words = BUFFER_SIZE_WORDS - frame_offset;
            do {
              dest[0] = ((src[0]>>current_bit_offset)|(src[-1]<<(16-current_bit_offset)));
              ++dest;
              ++src;
            } while( --words );
          }
          partial_buffer = 1;
        }
        ++incoming_count;
        if( ++incoming_tail == NUM_INCOMING_BUFFERS )
          incoming_tail = 0;
      } else {
        overflow = 1;
      } 
    }
  }

  if( write_to_flash ) {
    DataEEWrite(stored_delay_in_ms);
    write_to_flash = 0;
  }

  current_bit_offset = calc_bit_offset(RxHoldingBuffer);
  semilocked = (current_bit_offset > 0);
  if( current_bit_offset == 0xFFFF ) {
    locked = 0;
    partial_buffer = 0;
    block_offset = 0;
  } else {
    if( !locked || ++block_offset >= 12 ) {
      if( current_bit_offset == 0 )
        first_word = RxHoldingBuffer[0]&0xFF00;
      else
        first_word = ((RxHoldingBuffer[0]>>current_bit_offset)|(last_received_word<<(16-current_bit_offset)))&0xFF00;
      if( first_word == PREAMBLE_Z || first_word == (PREAMBLE_Z^0xFF00) ) {
        locked = 1;
        block_offset = 0;
        frame_offset = 0;
      } else {
        unsigned char i;
        unsigned short value;
        locked = 0;
        for( i = 8; i < sizeof(RxBufferA)/2; i += 8 ) {
          if( current_bit_offset == 0 )
            value = RxHoldingBuffer[i]&0xFF00;
          else
            value = ((RxHoldingBuffer[i]>>current_bit_offset)|(RxHoldingBuffer[i-1]<<(16-current_bit_offset)))&0xFF00;
          if( value == PREAMBLE_Z || value == (PREAMBLE_Z^0xFF00) ) {
            locked = 1;
            block_offset = 0;
            frame_offset = i;
            break;
          }
        }
        if( i == sizeof(RxBufferA)/2 ) {
          locked = 0;
          partial_buffer = 0;
        }
      }
    }
  }

  last_received_word = RxHoldingBuffer[BUFFER_SIZE_WORDS-1];
  last_spdif_timer_index = (last_spdif_timer_index+1)&3;
  last_spdif_timer[last_spdif_timer_index] = timer;
}


//// Main program ////

#ifdef DEBUG
unsigned short debug_timer;
#endif

int main (void) {
  unsigned short write_pos = 0, read_pos = 0, head_pos = 0, new_delay_in_ms = 0;
  unsigned short underflowed = 0, ir_led_on = 0, led_flash_timer = 0;
  unsigned char green_on, yellow_on;

  setup_clock();
  DataEEInit();
  stored_delay_in_ms = DataEERead();
  if( stored_delay_in_ms > 1500 )
    stored_delay_in_ms = 200;
  new_delay_in_ms = delay_in_ms = stored_delay_in_ms;
  generate_biphase_tables();
  sram_init();
  setup_spdif_in_timer();
  setup_dci();
  init_ir();

  IPC3bits.DMA1IP = 6;
  IPC1bits.DMA0IP = 6;
  IPC15bits.DCIIP = 4;
  IPC2bits.SPI1IP = 3;
  IPC1bits.T2IP = 2;
  IPC4bits.CNIP = 1;

  TRISBbits.TRISB3 = 0;
  TRISBbits.TRISB15 = 0;

  if( !memory_ok() ) {
    LATBbits.LATB15 = 0;
    while(1) {
      LATBbits.LATB3 = !LATBbits.LATB3;
      delay_ms(250);
    }
  }

  while(1) {
    if( !locked ) {
      head_pos = write_pos;
      new_delay_in_ms = delay_in_ms;
      spdif_rate_change = 0;
      if( !bypass_mode ) {
        memset(TxBufferA, 0, sizeof(TxBufferA));
        memset(TxBufferB, 0, sizeof(TxBufferB));
      }
    }
    if( incoming_count > 0 && outgoing_count < NUM_OUTGOING_BUFFERS && !bypass_mode ) {
	  unsigned char first = (incoming[incoming_head][0]&0xFF00) == PREAMBLE_Z || ((~incoming[incoming_head][0])&0xFF00) == PREAMBLE_Z;
#ifdef DEBUG
      debug_timer += outgoing_count; // this gives a visual indication of how full the buffers are
#endif

      decode_data(DecodedDataBuffer, (const unsigned char*)incoming[incoming_head]);
      if( ++incoming_head == NUM_INCOMING_BUFFERS )
        incoming_head = 0;
      asm volatile("disi #3");
      --incoming_count;

      if( spdif_rate_change == 3 ) {
        head_pos = write_pos;
        new_delay_in_ms = delay_in_ms;
		spdif_rate_change = 0;
      }

      sram_write_data(write_pos*7*(BUFFER_SIZE_WORDS/64), (const unsigned char*)DecodedDataBuffer, sizeof(DecodedDataBuffer));
      if( first )
  	    block_states[write_pos>>4] |=  (1<<(write_pos&15));
      else
  	    block_states[write_pos>>4] &= ~(1<<(write_pos&15));
      first = (block_states[read_pos>>4] >> (read_pos&15)) & 1;
      if( locked && first && new_delay_in_ms ) {
        unsigned short i, temp;
        if( new_delay_in_ms ) {
          delay_in_ms = new_delay_in_ms;
          new_delay_in_ms = 0;
          delay_changed_counter_high = 1;
        }

        read_pos = write_pos + BUFFERS_IN_SRAM - calculate_delay_blocks(delay_in_ms);
        if( read_pos >= BUFFERS_IN_SRAM )
          read_pos -= BUFFERS_IN_SRAM;
        temp = read_pos;
        for( i = 0; i < 32; ++i, ++temp ) {
          if( temp == BUFFERS_IN_SRAM )
            temp = 0;
          if( (block_states[temp>>4] >> (temp&15)) & 1 ) {
            read_pos = temp;
            break;
          }  
        }
      }
      sram_read_data((unsigned char*)DecodedDataBuffer, read_pos*7*(BUFFER_SIZE_WORDS/64), sizeof(DecodedDataBuffer));
      if( ++write_pos == BUFFERS_IN_SRAM )
        write_pos = 0;
      if( ++read_pos == BUFFERS_IN_SRAM )
        read_pos = 0;
      if( head_pos == 0xFFFF ) {
        encode_data((unsigned char*)outgoing[outgoing_tail], DecodedDataBuffer, first);
      } else {
        memset(outgoing[outgoing_tail], 0, sizeof(outgoing[outgoing_tail]));
        if( read_pos == head_pos )
          head_pos = 0xFFFF;
      }
      if( ++outgoing_tail == NUM_OUTGOING_BUFFERS )
        outgoing_tail = 0;
      asm volatile("disi #3");
      ++outgoing_count;
      if( outgoing_count < NUM_OUTGOING_BUFFERS && head_pos == 0xFFFF ) {
        memset(outgoing[outgoing_tail], 0, sizeof(outgoing[outgoing_tail]));
        if( ++outgoing_tail == NUM_OUTGOING_BUFFERS )
          outgoing_tail = 0;
        asm volatile("disi #3");
        ++outgoing_count;
      }
    } else if( ir_final_code ) {
      ir_led_on = 0xFFFF-4096;
      unsigned long code = ir_final_code;
      ir_final_code = 0;
      switch(code) {
      case RC5_A1012_VCR_1|IR_RC5:
      case RC5_A1012_VCR_2|IR_RC5:
      case RC5_A1012_VCR_3|IR_RC5:
      case RC5_A1012_VCR_4|IR_RC5:
      case RC5_A1012_VCR_5|IR_RC5:
      case RC5_A1012_VCR_6|IR_RC5:
      case RC5_A1012_VCR_7|IR_RC5:
      case RC5_A1012_VCR_8|IR_RC5:
      case RC5_A1012_VCR_9|IR_RC5:
        new_delay_in_ms = (code + 1 - RC5_A1012_VCR_1) * 100;
        break;
      case RC5_A1012_VCR_0|IR_RC5:
        new_delay_in_ms = 1000;
        break;
      case RC5_A1012_VCR_CHUP|IR_RC5:
      case RC5_A1012_VCR_CHUP|IR_RC5|IR_REPEAT:
        if( delay_in_ms < 1500 ) {
          if( delay_in_ms < 1400 )
            new_delay_in_ms = delay_in_ms + 100;
          else
            new_delay_in_ms = 1500;
        }
        break;
      case RC5_A1012_VCR_CHDN|IR_RC5:
      case RC5_A1012_VCR_CHDN|IR_RC5|IR_REPEAT:
        if( delay_in_ms > 10 ) {
          if( delay_in_ms > 110 )
            new_delay_in_ms = delay_in_ms - 100;
          else
            new_delay_in_ms = 10;
        }
        break;
      case RC5_A1012_VCR_VOLUP|IR_RC5:
      case RC5_A1012_VCR_VOLUP|IR_RC5|IR_REPEAT:
        if( delay_in_ms < 1500 ) {
          if( delay_in_ms < 1490 )
            new_delay_in_ms = delay_in_ms + 10;
          else
            new_delay_in_ms = 1500;
        }
        break;
      case RC5_A1012_VCR_VOLDN|IR_RC5:
      case RC5_A1012_VCR_VOLDN|IR_RC5|IR_REPEAT:
        if( delay_in_ms > 10 ) {
          if( delay_in_ms > 20 )
            new_delay_in_ms = delay_in_ms - 10;
          else
            new_delay_in_ms = 10;
        }
        break;
      case RC5_A1012_VCR_MUTE|IR_RC5:
      case RC5_AR1729_VCR_MUTE|IR_RC5:
        bypass_mode ^= 1;
        if( bypass_mode == 0 ) {
          head_pos = write_pos;
          new_delay_in_ms = delay_in_ms;
        }
        break;
      }
    } else if( delay_changed_counter_high ) {
      if( ++delay_changed_counter_low == 0 ) {
        if( ++delay_changed_counter_high == 1024 ) {
          if( delay_in_ms != stored_delay_in_ms ) {
            write_to_flash = 1;
            stored_delay_in_ms = delay_in_ms;
            delay_changed_counter_high = 0;
          }
        }
      }
    }

    if( underflow || overflow ) {
      underflowed = 1;
      asm("disi #16");
      if( outgoing_count < 4 - incoming_count ) {
        // this should never happen and it doesn't seem to, but putting in code to handle in Just In Case (tm)
        memset(outgoing[outgoing_tail], 0, sizeof(outgoing[outgoing_tail]));
        if( ++outgoing_tail == NUM_OUTGOING_BUFFERS )
          outgoing_tail = 0;
        asm volatile("disi #3");
        ++outgoing_count;
      } else {
        underflow = 0;
        overflow = 0;
      }
    }

	green_on = !bypass_mode && (!locked || head_pos == 0xFFFF || !(led_flash_timer&32768));
#ifdef DEBUG
    green_on ^= ((debug_timer>>10)&1); // this gives a visual indication of how full the buffers are
#endif
    yellow_on = ((locked && !underflowed) || (!locked && bypass_mode && (led_flash_timer < 1024))) ^ (ir_led_on ? 1 : 0);
    
    if( underflowed )
      ++underflowed;
    if( ir_led_on )
      ++ir_led_on;
    LATBbits.LATB15 = green_on;
    LATBbits.LATB3 = yellow_on;

    ++led_flash_timer;
  }
  return 0;
}
